#include "DepthOfField.hlsli"

// Given a radius offset, convert to an offset in texture coordinate space
float2 TexOffsetTexS(float radius) {
	return float2(radius / FilmWidth, radius / FilmHeight);
}

float4 Blur(float2 texCoords) {
	// Sample the depth and convert to linear view space Z
	float depth = DepthStencilTex.SampleLevel(LinearSampler, texCoords, 0);
	float linearDepth = ProjectionB / (depth - ProjectionA);
	float CoCRadius = CalculateCoCRadiusFilmS(linearDepth);

	// Convert CoC radius from film space (meters) to image space (in pixel width) to mip level
	float CoCImageS = CoCRadius * ImageWidth / FilmWidth;
	float mipLevel = log2(CoCImageS);

	return RenderTargetTex.SampleLevel(LinearSampler, texCoords, mipLevel);
}

float4 BlurOld(float3 viewRay, float2 texCoords) {
	float3 centerPos = PositionFromDepthViewS(viewRay, texCoords);

	float4 result = RenderTargetTex.SampleLevel(LinearSampler, texCoords, 0);
	float sampleWeight = 1;

	// DoF test by sampling depth and calculating CoC for nearby pixels.
	static const int numOffsets = 16;
	static const float2 offsets[] = {
		float2(-0.166,  0.266),
		float2( 0.947,  0.099),
		float2( 0.218, -0.685),
		float2(-0.135, -0.450),
		float2(-0.830, -0.522),
		float2(-0.846,  0.073),
		float2( 0.284,  0.657),
		float2( 0.907,  0.263),
		float2(-0.757,  0.100),
		float2(-0.353, -0.519),
		float2(-0.080,  0.449),
		float2(-0.559, -0.550),
		float2( 0.684, -0.190),
		float2(-0.544,  0.410),
		float2(-0.517, -0.644),
		float2( 0.413,  0.740),
		float2( 0.219, -0.716),
		float2( 0.018, -0.657),
		float2(-0.720, -0.139),
		float2( 0.536,  0.094),
		float2( 0.863,  0.125),
		float2( 0.192,  0.106),
		float2( 0.810,  0.298),
		float2( 0.314, -0.264),
		float2( 0.395,  0.416),
		float2(-0.206,  0.361),
		float2(-0.913, -0.256),
		float2(-0.410,  0.548),
		float2( 0.647,  0.017),
		float2( 0.914,  0.347),
		float2(-0.413, -0.441),
		float2(-0.172,  0.377),
	};

	int i;
	float searchRadius = kCoCSearchRadius;
	for(i = 0; i < numOffsets; ++i) {
		float2 curOffset = offsets[i] * TexOffsetTexS(searchRadius);
		float2 curCoords = texCoords + curOffset;

		// Perform depth and radius test to make sure we occlude and blur properly.
		float3 curPos = PositionFromDepthViewS(viewRay, curCoords);
		if(curPos.z >= centerPos.z) {

			// Current sample is in front, check its CoC radius.
			float CoCRadius = CalculateCoCRadiusFilmS(curPos.z);
			float curRadius = searchRadius * length(offsets[i]);
			if(curRadius < CoCRadius) {

				// Current sample will contribute.
				float curWeight = 1;
				result += RenderTargetTex.SampleLevel(LinearSampler, curCoords, 0) * curWeight;
				sampleWeight += curWeight;
			}
		}
	}

	return result / sampleWeight;
}

// Output depth values in a range of [0, 1] for debug rendering
float4 RenderDepth(float3 viewRay, float2 texCoords) {
	float3 posViewS = PositionFromDepthViewS(viewRay, texCoords);
	float depth = posViewS.z / FarClip;
	return float4(depth, depth, depth, 1);
}

float4 RenderCoCSize(float3 viewRay, float2 texCoords) {
	float3 posViewS = PositionFromDepthViewS(viewRay, texCoords);
	float cocRadius = CalculateCoCRadiusFilmS(posViewS.z);
	float cocSize = cocRadius  / kCoCSearchRadius; // Render max diameter as white
	return float4(cocSize, cocSize, cocSize, 1);
}